<?php

namespace App\Controllers;

use App\Core\Controller;
use App\Core\Database;
use App\Models\Reservation;

/**
 * CalendarController – správa kalendáře zimního stadionu
 */
class CalendarController extends Controller
{
    private Database $db;

    public function __construct(Database $db)
    {
        parent::__construct();
        $this->db = $db;
    }

    /**
     * Zobrazí kalendář s rezervacemi pro daný den
     * GET: ?date=2024-02-09 (nebo aktuální den)
     */
    public function index()
    {
        $date = $_GET['date'] ?? date('Y-m-d');

        $reservationModel = new Reservation($this->db);
        $reserved = $reservationModel->getOccupiedHours($date);
        $reservations = $reservationModel->getByDate($date);

        $this->view->render('calendar/index', [
            'date' => $date,
            'reserved' => $reserved,
            'reservations' => $reservations
        ]);
    }

    /**
     * Zobrazí formulář pro novou rezervaci
     */
    public function create()
    {
        $date = $_GET['date'] ?? date('Y-m-d');

        $reservationModel = new Reservation($this->db);
        $reserved = $reservationModel->getOccupiedHours($date);

        $this->view->render('calendar/form', [
            'date' => $date,
            'reserved' => $reserved,
            'mode' => 'create',
            'reservation' => null
        ]);
    }

    /**
     * Uloží novou rezervaci
     * POST: date, hour_start, hour_end, title
     */
    public function store()
    {
        $date = $_POST['date'] ?? null;
        $hour_start = (int)($_POST['hour_start'] ?? -1);
        $hour_end = (int)($_POST['hour_end'] ?? -1);
        $title = trim($_POST['title'] ?? '');

        if (!$date || $hour_start < 0 || $hour_end < 0 || !$title) {
            $_SESSION['error'] = 'Všechna pole jsou povinná!';
            $this->redirect("?action=create&date=$date");
        }

        $reservationModel = new Reservation($this->db);

        if ($reservationModel->create($date, $hour_start, $hour_end, $title)) {
            $_SESSION['success'] = 'Rezervace byla úspěšně vytvořena!';
            $this->redirect("?date=$date");
        } else {
            $_SESSION['error'] = 'Nelze vytvořit rezervaci! Možné příčiny: overlapping hodin nebo neplatný rozsah.';
            $this->redirect("?action=create&date=$date");
        }
    }

    /**
     * Zobrazí formulář pro editaci rezervace
     */
    public function edit()
    {
        $id = (int)($_GET['id'] ?? 0);

        if (!$id) {
            $this->redirect('/');
        }

        $reservationModel = new Reservation($this->db);
        $reservation = $reservationModel->getById($id);

        if (!$reservation) {
            $_SESSION['error'] = 'Rezervace nebyla nalezena!';
            $this->redirect('/');
        }

        $date = $reservation['date'];
        $reserved = $reservationModel->getOccupiedHours($date);

        $this->view->render('calendar/form', [
            'date' => $date,
            'reserved' => $reserved,
            'mode' => 'edit',
            'reservation' => $reservation
        ]);
    }

    /**
     * Aktualizuje existující rezervaci
     * POST: id, date, hour_start, hour_end, title
     */
    public function update()
    {
        $id = (int)($_POST['id'] ?? 0);
        $date = $_POST['date'] ?? null;
        $hour_start = (int)($_POST['hour_start'] ?? -1);
        $hour_end = (int)($_POST['hour_end'] ?? -1);
        $title = trim($_POST['title'] ?? '');

        if (!$id || !$date || $hour_start < 0 || $hour_end < 0 || !$title) {
            $_SESSION['error'] = 'Všechna pole jsou povinná!';
            $this->redirect("?action=edit&id=$id");
        }

        $reservationModel = new Reservation($this->db);

        if ($reservationModel->update($id, $date, $hour_start, $hour_end, $title)) {
            $_SESSION['success'] = 'Rezervace byla úspěšně aktualizována!';
            $this->redirect("?date=$date");
        } else {
            $_SESSION['error'] = 'Nelze aktualizovat rezervaci! Možné příčiny: overlapping hodin nebo neplatný rozsah.';
            $this->redirect("?action=edit&id=$id");
        }
    }

    /**
     * Smaže rezervaci
     * GET: id
     */
    public function delete()
    {
        $id = (int)($_GET['id'] ?? 0);

        if (!$id) {
            $this->redirect('/');
        }

        $reservationModel = new Reservation($this->db);
        $reservation = $reservationModel->getById($id);

        if (!$reservation) {
            $_SESSION['error'] = 'Rezervace nebyla nalezena!';
            $this->redirect('/');
        }

        $date = $reservation['date'];

        if ($reservationModel->delete($id)) {
            $_SESSION['success'] = 'Rezervace byla smazána!';
        } else {
            $_SESSION['error'] = 'Chyba při mazání rezervace!';
        }

        $this->redirect("?date=$date");
    }
}

