<?php
namespace App\Models\Entities;

use App\Core\Model;

/**
 * RezervaceEntity – entita pro rezervaci ledu
 * Reprezentuje jednu rezervaci v systému
 */
class RezervaceEntity extends Model
{
    public ?int $id = null;
    public ?string $date = null;
    public ?int $hour_start = null;
    public ?int $hour_end = null;
    public string $title = '';
    public ?string $created_at = null;

    /**
     * Vrátí čas začátku jako formátovaný string (HH:00)
     */
    public function getStartTimeFormatted(): string
    {
        return str_pad($this->hour_start ?? 0, 2, '0', STR_PAD_LEFT) . ':00';
    }

    /**
     * Vrátí čas konce jako formátovaný string (HH:00)
     */
    public function getEndTimeFormatted(): string
    {
        return str_pad($this->hour_end ?? 0, 2, '0', STR_PAD_LEFT) . ':00';
    }

    /**
     * Vrátí čas jako rozsah (HH:00 – HH:00)
     */
    public function getTimeRange(): string
    {
        return $this->getStartTimeFormatted() . ' – ' . $this->getEndTimeFormatted();
    }

    /**
     * Vrátí datum jako formátovaný string
     */
    public function getDateFormatted(): string
    {
        return date('d. m. Y (l)', strtotime($this->date ?? 'now'));
    }

    /**
     * Vrátí čas vytvoření jako formátovaný string
     */
    public function getCreatedAtFormatted(): string
    {
        return date('H:i', strtotime($this->created_at ?? 'now'));
    }

    /**
     * Ověří, zda je daná hodina obsažena v rezervaci
     */
    public function containsHour(int $hour): bool
    {
        return $hour >= $this->hour_start && $hour < $this->hour_end;
    }

    /**
     * Vrátí délku rezervace v hodinách
     */
    public function getDuration(): int
    {
        return ($this->hour_end ?? 0) - ($this->hour_start ?? 0);
    }
}
